module.exports = function (grunt) {
	'use strict';
	var async = require('async'),
		pkg = grunt.file.readJSON('package.json'),
		publishRelease = function (err, stdout, stderr, cb) {
			if (!err) {
				grunt.config.set('artifactory.release.options', {
					publish: [
						{
							id: 'com.agilex.healthcare.portal.js:' + pkg["artifact-name"] + ':tgz',
							version: pkg.version + '.' + stdout.trim(),
							path: 'build/'
						}
					]
				});
				grunt.task.run('artifactory:release:publish');
			} else {
				console.log('Command to get Git commit count failed');
				console.log(err);
				console.log(stderr);
			}
			cb();
		};
	grunt.initConfig({
		pkg: grunt.file.readJSON('package.json'),
		jshint: { // configure the task
			// lint your project's server code
			options: {laxcomma: true},
			gruntfile: {
				src: [
					'Gruntfile.js'
				],
				directives: {
					node: true
				},
				options: {
				}
			},
			server: {  // could also specify 'client'
				src: [
					'**/*.js', '!**/*.min.js', '!test/libs/**/*.js', '!node_modules/**/*.js', '!runtime/**/*', '!test/runtime/**/*', '!build/**/*', '!core/extensions/**'
				],
				directives: {
					browser: true,  // from example for client
					node: true,     // from example for server
					globals: {
						// 'jQuery' etc.
					}
				},
				options: {
					expr: true
				}
			}
		},
		jasmine: {
			app: {
				options: {
					outfile: "_SpecRunner.html",
					junit: {
						path: "test/build/junit-reports"
					},
					specs: "test/specs/*.js",
					template: require('grunt-template-jasmine-requirejs'),
					templateOptions: {
						requireConfigFile: ['config.js' ]
					}
				}
			}
		},
		jade: {
			compile: {
				options: {
					data: {
						debug: false
					},
					pretty: false
				},
				files: [
					{
						expand: true,
						cwd: 'templates/',
						src: ['**/*.jade'],
						dest: 'templates/',
						ext: '.html'
					}
				]
			}
		},
		compass: {
			dist: {
				options: {
					sassDir: '_assets/sass',
					cssDir: '_assets/css'
				}
			}
		},
		autoprefixer: {
			all: {
				options: {
					browsers: ['last 2 version']
				},
				files: [
					{
						expand: true,
						cwd: '_assets/',
						src: ['css/*.css'],
						dest: "_assets/",
						ext: '.css'
					}
				]
			}
		},
		shell: {
			publishRelease: {
				command: 'git rev-list HEAD --count',
				options: {
					callback: publishRelease
				}
			},
			versionHash: {
				"command": "git rev-parse HEAD",
				options: {
					callback: function (err, stdout, stderr, cb) {
						console.log("hash");
						console.log(stdout.trim());
						console.log("end of hash");
						var originalVersion = grunt.config.get("pkg.version");
						grunt.config.set("pkg.version", originalVersion + "_" + stdout.trim());
						grunt.task.run(["deploy"]);
						//grunt.config.set("pkg.version", originalVersion );
						cb();
					}
				}
			},
			cleanHTML: {
				command: 'rm *.html',
				options: {
					execOptions: {
						cwd: 'templates'
					}
				}
			}
		},
		watch: {
			gruntfile: {
				files: 'Gruntfile.js',
				tasks: ['jshint:gruntfile']
			},
			compass: {
				files: ['_assets/sass/*.scss'],
				tasks: ['compass']
			},
			jade: {
				files: ["templates/**/*.jade"],
				tasks: ['jade']
			},
			scripts: {
				files: ['core/**/*.js', '!test/libs/**/*.js'],
				tasks: ['jshint:server'],
				options: {
					spawn: false
				}
			},
			runtime: {
				tasks: ['buildTestRuntime', 'tomcat_redeploy'],
				files: ['*.js', 'core/**/*.js', '!test/libs/**/*.js', 'templates/**/*.html', '_assets/css/*.css', '_assets/img/*']
			},
			livereload: {
				options: {
					livereload: true
				},
				files: [
					'*.js',
					'js/**/*.js',
					'css/*.css'
				]
			}
		},
		copy: {
			appletToRuntime: {
				src: ['_assets/css/*.css', 'core/**', 'loader.js', 'Controller.js', 'util.js', 'templates/*.html', '_assets/img/*'],
				dest: 'test/runtime/container/applets/' + pkg['artifact-name'] + '/'
			}
		},
		compress: {
			main: {
				options: {
					mode: 'tgz',
					archive: 'build/skin-check.tar.gz'
				},
				files: [
					// exclude node_modules/, build/, sass/, SpecRunner.html, js/test
					{ expand: true, src: ['../wound-care/**/*'] }
				]
			},
			runtimeWar: {
				options: {
					archive: 'build/' + pkg['artifact-name'] + '.war',
					mode: 'zip'
				},
				files: [
					{
						cwd: 'test/runtime/', expand: true, src: ['**/*', '!**/*.tgz']
					}
				]
			}
		},
		artifactory: {
			options: {
				url: 'https://ihealthbuild.agilexhealth.com',
				repository: 'js-local',
				username: 'admin',
				password: 'Agilexadmin99$'
			},
			snapshot: {
				files: [
					{
						src: ['**/*', '!node_modules/**/*', '!build/**/*', '!test/**/*', '!*SpecRunner.html', '!README.md', '!.gitignore', '!config.js', '!package.json', '!Gruntfile.js', '!**/*.jade', '!**/*.scss', '!gems/**/*', '!**/*.gem']
					}
				],
				options: {
					publish: [
						{
							id: 'com.agilex.healthcare.portal.js:<%= pkg["artifact-name"] %>:tgz',
							version: '<%= pkg["version"] %>',
							path: 'build/'
						}
					]
				}
			},
			release: {
				files: [
					{
						src: ['**/*', '!node_modules/**/*', '!build/**/*', '!test/**/*', '!*SpecRunner.html', '!README.md', '!.gitignore', '!config.js', '!package.json', '!Gruntfile.js', '!**/*.jade', '!**/*.scss', '!gems/**/*', '!**/*.gem']
					}
				]
			}
		},
		tomcat_deploy: {
			host: 'localhost',
			login: 'admin',
			password: 'Agilexadmin99$',
			path: '/' + pkg['artifact-name'],
			port: 8080,
			dist: 'test/runtime',
			deploy: '/manager/deploy',
			undeploy: '/manager/undeploy'
		}
	});

	var checkMysql = function (done) {
		var mysql = require('mysql');
		var connection = mysql.createConnection({
			host: 'localhost',
			user: 'root',
			password: 'Agilexadmin99$'
		});
		connection.connect(function (err) {
			// connected! (unless `err` is set)
			grunt.log.writeln("testing MYSQL");
			if (err) {
				grunt.log.error(["MYSQL might not be running", err]);
			} else {
				grunt.log.ok("MYSQL: connection OK");
			}
			done();
		});
	};
	var checkMongo = function (done) {
		var MongoClient = require('mongodb').MongoClient;

		MongoClient.connect('mongodb://127.0.0.1:27017/', {}, function (err, db) {
			grunt.log.writeln("testing MongoDB");
			if (err) {
				grunt.log.error(["MongoDB might not be running", err]);
			} else {
				grunt.log.ok("MongoDB: connection OK");
				// Let's close the db
				db.close();
			}
			done();
		});
	};
	var checkTomcat = function (done) {
		var http = require('http'),
			options = {
				host: 'localhost',
				port: 8080,
				path: '/'
			};

		http.get(options,function (res) {
			grunt.log.writeln("testing Tomcat");
//			console.log("Got response: " + res.statusCode);
			if (res.statusCode === 200) {
				grunt.log.ok("tomcat: connection OK");
			} else {
				grunt.log.error(["Tomcat responded with status code: " + res.statusCode]);
			}
			done();
		}).on('error', function (e) {
				grunt.log.writeln("testing Tomcat");
				grunt.log.error(["Tomcat might not be working: " + e.message]);
				done();
			});
	};
	var checkDeployments = function (done) {
		done();
	};
	grunt.task.registerTask("WTF",
		"this task will try to help you figure if there is a service that is not running",
		function () {
			var gruntDone = this.async();
			async.parallel([
				checkMysql ,
				checkMongo ,
				checkTomcat ,
				checkDeployments
			], function () {
				gruntDone();
			});
		});

	grunt.registerTask('buildTestRuntime', 'Build applet runtime for testing', function () {

		var appletName,
			appletVersion,
			appletPath,
			runtimePkg = grunt.file.readJSON('test/runtime/package.json'),
			excludeList = ['Portal', 'backbone', 'marionette', 'jquery', 'underscore', 'jquerymobile', 'hbs', 'json2', 'i18nprecompile'],
			appletListLength = runtimePkg.applets.length,
			fetchList = [],
			dynamicallyLoadedModules = [],
			excludeShallowList = [];
		// dynamically create require js modules and dependency fetch list based on applets defined in package.json
		for (var i = 0; i < appletListLength; i += 1) {
			appletName = runtimePkg.applets[i].name;
			appletVersion = runtimePkg.applets[i].version;
			appletPath = "applets/" + appletName + "/loader";

			excludeShallowList.push(appletPath);

			dynamicallyLoadedModules.push({
				name: appletPath,
				exclude: excludeList
			});

			fetchList.push({
				id: 'com.agilex.healthcare.portal.js:' + appletName + ':tgz:' + appletVersion,
				path: 'test/runtime/container/applets/' + appletName
			});
		}

		// add init require js module
		dynamicallyLoadedModules.unshift({
			name: "core/init",
			excludeShallow: excludeShallowList
		});

		// add portal container dependency
		fetchList.unshift({
			id: 'com.agilex.healthcare.portal.js:portal-container:tgz:' + runtimePkg["portal-container"].version,
			path: 'test/runtime/container/'
		});


		grunt.config.set('artifactory.runtime.options', {
			fetch: fetchList
		});

		grunt.task.run('artifactory:runtime:fetch');
		grunt.task.run('build');
		grunt.task.run('copy:appletToRuntime');


		//add the applet under development so its recognized when app loaded
		runtimePkg.applets[appletListLength] = {"name": pkg['artifact-name'], "version": pkg.version};
		grunt.file.write('test/runtime/version.json', JSON.stringify({
			"name": runtimePkg.name,
			"title": runtimePkg.title,
			"version": runtimePkg.version,
			"artifact-name": runtimePkg['artifact-name'],
			"uber-portal": runtimePkg['uber-portal'],
			"applets": runtimePkg.applets,
			"links": runtimePkg.links,
			"widgets": runtimePkg.widgets,
			"description": runtimePkg.description
		}));

		grunt.task.run('compress:runtimeWar');

	});

	grunt.loadNpmTasks('grunt-contrib-jshint');
	grunt.loadNpmTasks('grunt-contrib-compass');
	grunt.loadNpmTasks('grunt-contrib-copy');
	grunt.loadNpmTasks('grunt-contrib-jade');
	grunt.loadNpmTasks('grunt-contrib-jasmine');
	grunt.loadNpmTasks('grunt-contrib-watch');
	grunt.loadNpmTasks('grunt-contrib-compress');
	grunt.loadNpmTasks('grunt-contrib-jshint');
	grunt.loadNpmTasks('grunt-shell');
	grunt.loadNpmTasks('grunt-tomcat-deploy');
	grunt.loadNpmTasks('grunt-artifactory-artifact');
	grunt.loadNpmTasks('grunt-autoprefixer');

	grunt.registerTask('test', [ 'jasmine' ]);
	grunt.registerTask('build', [ 'jshint', 'compass', 'autoprefixer:all', 'jade', 'test']);
	grunt.registerTask('runtime', ['buildTestRuntime', 'tomcat_redeploy']);
	grunt.registerTask('deploy', [ 'build', 'artifactory:snapshot:publish' ]);
	grunt.registerTask('release', [ 'deploy', 'shell:publishRelease' ]);
	grunt.registerTask('default', [ 'build', 'test', 'watch' ]);
	grunt.registerTask('snapshot', ["shell:versionHash"]);
	grunt.registerTask('copyToTomcat', [ 'copy:appletToRuntime', 'tomcat_redeploy']);
};
